/*
 * This file demonstrates an attack on 1 full round of AES-128 with one known plaintext
 * (Note : 1 full round = ARK o MC o SR o SB o ARK) 
 *
 * The attack is a guess-and-determine attack in which 5 bytes are enumerated.
 * Thus the attack perform 2^40 iterations of a simple loop.
 *
 * The attack was described and published in 
 * "Low Data Complexity Attacks on AES"
 * by Charles Bouillaguet, Patrick Derbez, Orr Dunkelman, Nathan Keller, Pierre-Alain Fouque and Vincent Rijmen
 * available at http://eprint.iacr.org/2010/633
 *
 * The attack was "automatically" generated by the software tool described in a subsequent paper
 *
 * The actual attack procedure is in the file "attack_(par/seq).c". 
 * These are... the sequential and parallel versions of the attack. 
 * The parallel version is just the sequential version with some OpenMP directive and a little reorganisation
 *
 * The AES implementation is adapted from rijndael-alg-ref.c v2.0 (August '99) by Charles Bouillaguet
 * this file was the Rijndael Reference ANSI C code
 * authors: Paulo Barreto
 *          Vincent Rijmen
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "rijndael.h"

#define SC	((BC - 4))

#include "boxes-ref.h"


static void KeyAddition(word8 a[4][MAXBC], word8 rk[4][MAXBC], word8 BC) {
	/* Exor corresponding text input and round key input bytes
	 */
	int i, j;
	
	for(i = 0; i < 4; i++)
   		for(j = 0; j < BC; j++) a[i][j] ^= rk[i][j];
}

static void ShiftRow(word8 a[4][MAXBC], word8 d, word8 BC) {
	/* Row 0 remains unchanged
	 * The other three rows are shifted a variable amount
	 */
	word8 tmp[MAXBC];
	int i, j;
	
	for(i = 1; i < 4; i++) {
	  //	  printf("Offset : %d\n", shifts[SC][i][d]);
		for(j = 0; j < BC; j++) tmp[j] = a[i][(j + shifts[SC][i][d]) % BC];
		for(j = 0; j < BC; j++) a[i][j] = tmp[j];
	}
}

static void Substitution(word8 a[4][MAXBC], const word8 box[256], word8 BC) {
	/* Replace every byte of the input by the byte at that place
	 * in the nonlinear S-box
	 */
	int i, j;
	
	for(i = 0; i < 4; i++)
		for(j = 0; j < BC; j++) a[i][j] = box[a[i][j]] ;
}
   
static void MixColumn(word8 a[4][MAXBC], word8 BC) {
        /* Mix the four bytes of every column in a linear way
	 */
	word8 b[4][MAXBC];
	int i, j;
		
	for(j = 0; j < BC; j++)
		for(i = 0; i < 4; i++)
			b[i][j] = mul(2,a[i][j])
				^ mul(3,a[(i + 1) % 4][j])
				^ a[(i + 2) % 4][j]
				^ a[(i + 3) % 4][j];
	for(i = 0; i < 4; i++)
		for(j = 0; j < BC; j++) a[i][j] = b[i][j];
}

static void InvMixColumn(word8 a[4][MAXBC], word8 BC) {
        /* Mix the four bytes of every column in a linear way
	 * This is the opposite operation of Mixcolumn
	 */
	word8 b[4][MAXBC];
	int i, j;
	
	for(j = 0; j < BC; j++)
	for(i = 0; i < 4; i++)             
		b[i][j] = mul(0xe,a[i][j])
			^ mul(0xb,a[(i + 1) % 4][j])                 
			^ mul(0xd,a[(i + 2) % 4][j])
			^ mul(0x9,a[(i + 3) % 4][j]);                        
	for(i = 0; i < 4; i++)
		for(j = 0; j < BC; j++) a[i][j] = b[i][j];
}

int rijndaelKeySched (word8 k[4][MAXKC], int keyBits, int blockBits, word8 W[MAXROUNDS+1][4][MAXBC]) {
	/* Calculate the necessary round keys
	 * The number of calculations depends on keyBits and blockBits
	 */
	int KC, BC, ROUNDS;
	int i, j, t, rconpointer = 0;
	word8 tk[4][MAXKC];   

	BC = blockBits/32;
	KC = keyBits/32;
	ROUNDS = 6 + (BC > KC ? BC : KC);

	
	for(j = 0; j < KC; j++)
		for(i = 0; i < 4; i++)
			tk[i][j] = k[i][j];
	t = 0;
	/* copy values into round key array */
	for(j = 0; (j < KC) && (t < (ROUNDS+1)*BC); j++, t++)
		for(i = 0; i < 4; i++) W[t / BC][i][t % BC] = tk[i][j];
		
	while (t < (ROUNDS+1)*BC) { /* while not enough round key material calculated */
		/* calculate new values */
		for(i = 0; i < 4; i++)
		  tk[i][0] ^= S[tk[(i+1)%4][KC-1]];
		tk[0][0] ^= rcon[rconpointer++];

		if (KC <= 6)
		  for(j = 1; j < KC; j++)
		    for(i = 0; i < 4; i++) tk[i][j] ^= tk[i][j-1];
		else {
		  for(j = 1; j < 4; j++)
		    for(i = 0; i < 4; i++) tk[i][j] ^= tk[i][j-1];
		  for(i = 0; i < 4; i++) tk[i][4] ^= S[tk[i][3]];
		  for(j = 5; j < KC; j++)
		    for(i = 0; i < 4; i++) tk[i][j] ^= tk[i][j-1];
		}
	/* copy values into round key array */
	for(j = 0; (j < KC) && (t < (ROUNDS+1)*BC); j++, t++)
		for(i = 0; i < 4; i++) W[t / BC][i][t % BC] = tk[i][j];
	}		

	return 0;
}
      
int rijndaelEncrypt (word8 a[4][MAXBC], int keyBits, int blockBits, word8 rk[MAXROUNDS+1][4][MAXBC])
{
	/* Encryption of one block. 
	 */
	int r, BC, ROUNDS;

	BC = blockBits/32;
	ROUNDS = 6 + (keyBits >= blockBits ? keyBits : blockBits)/32;

	/* begin with a key addition
	 */
	KeyAddition(a,rk[0],BC); 

        /* ROUNDS-1 ordinary rounds
	 */
	for(r = 1; r < ROUNDS; r++) {
		Substitution(a,S,BC);
		ShiftRow(a,0,BC);
		MixColumn(a,BC);
		KeyAddition(a,rk[r],BC);
	}
	
	/* Last round is special: there is no MixColumn
	 */
	Substitution(a,S,BC);
	ShiftRow(a,0,BC);
	KeyAddition(a,rk[ROUNDS],BC);

	return 0;
}   


void print_state(word8 a[4][MAXBC], int BC) {

  int i,j;
  for(i=0; i<4; i++) {
    for(j=0; j<BC; j++)
      printf("%02x ", a[i][j]);
    printf("\n");
  }
}

#include "attack_par.c"

int main() {

  word8 plaintext[4][4];
  word8 key[4][4];
  word8 expanded_key[MAXROUNDS+1][4][4];
  word8 ciphertext[4][4];

  int i,j;
  
  for(i=0; i<4; i++)
    for(j=0; j<4; j++) {
      key[i][j] = rand() & 0xff;
      ciphertext[i][j] = plaintext[i][j] = rand() & 0xff;
    }
      
  printf("Randomly-chosen Master Key : \n");
  print_state(key, 4);

  printf("Randomly-chosen plaintext : \n");
  print_state(plaintext, 4);

  // run the key-schedule
  rijndaelKeySched (key, 128, 128, expanded_key);
  //  rijndaelTest(plaintext, 128, 128, expanded_key);

  // make the ciphertext
  KeyAddition(ciphertext,expanded_key[0],4);

  //  printf("Actual X_0 : \n");
  //  print_state(ciphertext, 4);

  Substitution(ciphertext,S,4);
  ShiftRow(ciphertext,0,4);
  MixColumn(ciphertext,4);
  KeyAddition(ciphertext,expanded_key[1],4);

  printf("Corresponding ciphertext : \n");
  print_state(ciphertext, 4);

  word8 recovered_key[4][4];
  attack(plaintext, ciphertext, recovered_key);
  

  printf("\n\n");

}
